/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: DEV_InterfaceUser.c $
   Last Modification:
    $Author: $
    $Modtime:  $
    $Revision:  $

   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no

   Description:
     General example functions to initialize and configure a device      

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         16.08.2011  RM       Functions updated and reorganized

     1         17.03.2011  SD       initial version

**************************************************************************************/

#include "cifXHWFunctions.h"
#include "cifXEndianess.h"
/* Error & Type Def. */
#include "CIFXErrors.h"
#include "DEV_InterfaceUser.h"
#include "rcx_Public.h"

/* ============================================================================= */
/* Global infomation and definitions                                             */
/* ============================================================================= */
extern uint32_t ulSourceID;


/*****************************************************************************/
/*! deletes configuration
 *  example for use of put DEV_PutPacket() / DEV_GetPacket()
 * \param pChannel Channel instance
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t DEV_DeleteConfig(CHANNELINSTANCE* ptChannel)
{
  int32_t       lRet             = CIFX_NO_ERROR;
  CIFX_PACKET   tSendPacket      = {0};
  CIFX_PACKET   tReceivePacket   = {0};

  OS_Memset( &tSendPacket, 0, sizeof(tSendPacket));
  OS_Memset( &tReceivePacket, 0, sizeof(tReceivePacket));

  /* setup packet to delete configuration */
  tSendPacket.tHeader.ulDest = HOST_TO_LE32(RCX_PACKET_DEST_DEFAULT_CHANNEL);
  tSendPacket.tHeader.ulSrc  = HOST_TO_LE32(ulSourceID);
  tSendPacket.tHeader.ulLen  = HOST_TO_LE32(0);
  tSendPacket.tHeader.ulCmd  = HOST_TO_LE32(RCX_DELETE_CONFIG_REQ); /* delete configuration CMD */

  /* send packet */
  lRet = DEV_TransferPacket( ptChannel, &tSendPacket, &tReceivePacket, 
                             sizeof(tReceivePacket), CIFX_TO_SEND_PACKET, NULL, NULL);
  if ( (CIFX_NO_ERROR == lRet)                                    &&
       (CIFX_NO_ERROR == (lRet = LE32_TO_HOST(tReceivePacket.tHeader.ulState))) )

  {
    /* Activate the protocol stack withot a configuration */
    /* Processing channel init...            */
    lRet = DEV_DoChannelInit(ptChannel, CIFX_TO_FIRMWARE_START);

    if (CIFX_NO_ERROR == lRet)
    {
      int iCycleCount = 0;

      /* Update our flags after a channel init */
      /* read the host flags of the communication channel, first time to synchronise our internal status */
      DEV_ReadHostFlags( ptChannel, 0);

      /* Waiting for netX warmstart has finished */
      while ( !DEV_IsReady(ptChannel))
      {
        OS_Sleep(100);
        DEV_ReadHostFlags( ptChannel, 0);

        if (iCycleCount > 100)
        {
          lRet = CIFX_DEV_NOT_RUNNING;
          break;
        }

        iCycleCount++;
      }
    }
  }

  return lRet;
}

/*****************************************************************************/
/*! Process warmstart configuration
 * \param pChannel     Channel instance
 * \param ptSendPacket Pointer to a send packet buffer
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t DEV_ProcessWarmstart( CHANNELINSTANCE* ptChannel, CIFX_PACKET* ptSendPacket)
{
  int32_t     lRet             = CIFX_NO_ERROR;
  CIFX_PACKET tReceivePacket   = {0};

  OS_Memset( &tReceivePacket, 0, sizeof(tReceivePacket));
 
  /* check if packet is configured */
  if (!ptSendPacket->tHeader.ulCmd)
    return CIFX_INVALID_PARAMETER;

  /* send configuration Req packet */
  lRet = DEV_TransferPacket(  ptChannel, ptSendPacket, &tReceivePacket, 
                              sizeof(tReceivePacket), CIFX_TO_SEND_PACKET, NULL, NULL);
 
  /* check if we got an error within configuration packet */
  if ( (CIFX_NO_ERROR == lRet)                                    &&
       (CIFX_NO_ERROR == (lRet = LE32_TO_HOST(tReceivePacket.tHeader.ulState))) )
  {
    /* Activate the protocol stack with the new configuration */
    /* Processing channel init...            */
    lRet = DEV_DoChannelInit(ptChannel, CIFX_TO_FIRMWARE_START);
    
    if (CIFX_NO_ERROR == lRet)
    {
      int iCycleCount = 0;

      /* Update our flags after a channel init */
      /* read the host flags of the communication channel, first time to synchronise our internal status */
      DEV_ReadHostFlags( ptChannel, 0);

      /* Waiting for netX warmstart has finished */
      while ( !DEV_IsRunning(ptChannel))
      {
        OS_Sleep(100);
        DEV_ReadHostFlags( ptChannel, 0);

        if (iCycleCount > 100)
        {
          lRet = CIFX_DEV_NOT_RUNNING;
          break;
        }

        iCycleCount++;
      }
    }
  }

  return lRet;
}

/*****************************************************************************/
/*! Initialize internal data structures to use with DEV functions                           
 * \param ptDevInst          Buffer to the device instance structure
 * \param ulCommChannelCount Number of communication channels to init
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t DEV_Initialize(PDEVICEINSTANCE ptDevInst, uint32_t ulCommChannelCount)
{
  struct NETX_DEFAULT_DPM
  {
    NETX_SYSTEM_CHANNEL       SystemChannel;
    NETX_HANDSHAKE_CHANNEL    HandshakeChannel;
    NETX_DEFAULT_COMM_CHANNEL CommChannel[4];  

  } *ptDefaultDPM = (struct NETX_DEFAULT_DPM*)ptDevInst->pbDPM;

  int32_t           lRet         = CIFX_NO_ERROR;
  int               iIdx         = 0;
  PCHANNELINSTANCE  ptSysChannel = &ptDevInst->tSystemDevice;

  OS_Memset( ptSysChannel,   0, sizeof(*ptSysChannel));

  /* Initialize the globale device instance                                               */
  /* Physical address of the device                                                       */
  ptDevInst->ulPhysicalAddress = (uint32_t)ptDefaultDPM;
  /* Size of the DPM                                                                      */
  ptDevInst->pptCommChannels   = (PCHANNELINSTANCE*)OS_Memalloc(ulCommChannelCount * sizeof(*ptDevInst->pptCommChannels));
  
  if (HWIF_READ8( ptDevInst, ptDefaultDPM->SystemChannel.atChannelInfo[NETX_HANDSHAKE_CHANNEL_INDEX].tCom.bChannelType) == RCX_CHANNEL_TYPE_HANDSHAKE)
    ptDevInst->pbHandshakeBlock  = (uint8_t*)&ptDefaultDPM->HandshakeChannel;
  
  /* gloabal register block is only present on 64k DPM                                    */
  if(NETX_DPM_MEMORY_SIZE == ptDevInst->ulDPMSize)
  {  
    /* Setup pointer to global netX register block                                        */
    ptDevInst->ptGlobalRegisters = (PNETX_GLOBAL_REG_BLOCK)(ptDevInst->pbDPM +
                                                            ptDevInst->ulDPMSize -
                                                            sizeof(NETX_GLOBAL_REG_BLOCK));
  }

  for (iIdx = 0; iIdx < (int)ulCommChannelCount; iIdx++)
    ptDevInst->pptCommChannels[iIdx] = (PCHANNELINSTANCE)OS_Memalloc(sizeof(**ptDevInst->pptCommChannels));
  
  /* Initialize the system channel structure                                              */
  ptSysChannel->pvDeviceInstance    = ptDevInst;
  ptSysChannel->pbDPMChannelStart   = (uint8_t*)&ptDefaultDPM->SystemChannel;
  /* length of channel block                                                              */
  ptSysChannel->ulDPMChannelLength  = sizeof(ptDefaultDPM->SystemChannel);   

  /* ------------------------------------------------------------------------------------ */
  /* Mailbox specific data of the system channel                                          */
  /* ------------------------------------------------------------------------------------ */
  /* virtual start address of send mailbox                                                */
  ptSysChannel->tSendMbx.ptSendMailboxStart  = (NETX_SEND_MAILBOX_BLOCK*)&ptDefaultDPM->SystemChannel.tSystemSendMailbox;  
  ptSysChannel->tSendMbx.ulSendMailboxLength = NETX_SYSTEM_MAILBOX_MIN_SIZE;
  /* virtual start address of send mailbox                                                */
  ptSysChannel->tRecvMbx.ptRecvMailboxStart  = (NETX_RECV_MAILBOX_BLOCK*)&ptDefaultDPM->SystemChannel.tSystemRecvMailbox; 
  ptSysChannel->tRecvMbx.ulRecvMailboxLength = NETX_SYSTEM_MAILBOX_MIN_SIZE;
  /* Bitmask for Handshakeflags to send packet                                            */
  ptSysChannel->tSendMbx.ulSendCMDBitmask    = HSF_SEND_MBX_CMD;                                      
  /* Bitnumber for send packet flag (used for notification array indexing)                */
  ptSysChannel->tSendMbx.bSendCMDBitoffset   = HSF_SEND_MBX_CMD_BIT_NO;                               
  /* Bitnumber for recv packet ack flag (used for notification array indexing)            */
  ptSysChannel->tRecvMbx.ulRecvACKBitmask    = HSF_RECV_MBX_ACK;                                      
  /* Bitmask for Handshakeflags to ack recv. packet                                       */
  ptSysChannel->tRecvMbx.bRecvACKBitoffset   = HSF_RECV_MBX_ACK_BIT_NO;                               
  /* ------------------------------------------------------------------------------------ */

  /* Width of the handshake cell                                                          */
  ptSysChannel->bHandshakeWidth     = RCX_HANDSHAKE_SIZE_8BIT;                               
  /* pointer to channels handshake cell                                                   */
  ptSysChannel->ptHandshakeCell     = (NETX_HANDSHAKE_CELL*)&ptDefaultDPM->HandshakeChannel.tSysFlags;     
  /* !=0 if the channel instance belong to a systemdevice                                 */
  ptSysChannel->fIsSysDevice        = 1;                                                     
  /* channel lock */  
  ptSysChannel->pvLock              = OS_CreateLock();
  ptSysChannel->pvInitMutex         = OS_CreateMutex(); 

  /* Read actual Host state, in case they differ from 0 */
  DEV_ReadHostFlags(ptSysChannel, 1);
  DEV_ReadHandshakeFlags(ptSysChannel, 1, 0);  

  ptSysChannel->ulOpenCount         = 1;

  /* Initialize the communication channel structure                                       */
  for (iIdx = 0; iIdx < (int)ulCommChannelCount; iIdx++)
  {
    PCHANNELINSTANCE ptCommChannel = ptDevInst->pptCommChannels[iIdx];
    PIOINSTANCE      ptCommIn      = (PIOINSTANCE)OS_Memalloc(sizeof(*ptCommIn));
    PIOINSTANCE      ptCommOut     = (PIOINSTANCE)OS_Memalloc(sizeof(*ptCommOut));

    OS_Memset( ptCommChannel, 0, sizeof(*ptCommChannel));
    OS_Memset( ptCommIn, 0, sizeof(*ptCommIn));
    OS_Memset( ptCommOut, 0, sizeof(*ptCommOut));

    ptCommChannel->pvDeviceInstance    = ptDevInst;
    ptCommChannel->pbDPMChannelStart   = (uint8_t*)&ptDefaultDPM->CommChannel[iIdx];
    /* length of channel block                                                              */
    ptCommChannel->ulDPMChannelLength  = sizeof(NETX_DEFAULT_COMM_CHANNEL);    
    ptCommChannel->ulChannelNumber     = iIdx;
    ptCommChannel->ulBlockID           = iIdx + 2; /* BlockId0 = Sys ch, BlockId1 = Handshake ch */

    /* ------------------------------------------------------------------------------------ */
    /* Mailbox specific data of the communication channel                                   */
    /* ------------------------------------------------------------------------------------ */
    /* virtual start address of send mailbox                                                */
    ptCommChannel->tSendMbx.ptSendMailboxStart  = (NETX_SEND_MAILBOX_BLOCK*)&ptDefaultDPM->CommChannel[iIdx].tSendMbx;  
    ptCommChannel->tSendMbx.ulSendMailboxLength = NETX_CHANNEL_MAILBOX_SIZE;     
    /* virtual start address of send mailbox                                                */
    ptCommChannel->tRecvMbx.ptRecvMailboxStart  = (NETX_RECV_MAILBOX_BLOCK*)&ptDefaultDPM->CommChannel[iIdx].tRecvMbx;  
    ptCommChannel->tRecvMbx.ulRecvMailboxLength = NETX_CHANNEL_MAILBOX_SIZE;     
    /* Bitmask for Handshakeflags to send packet                                            */
    ptCommChannel->tSendMbx.ulSendCMDBitmask    = HCF_SEND_MBX_CMD;            
    /* Bitnumber for send packet flag (used for notification array indexing)                */
    ptCommChannel->tSendMbx.bSendCMDBitoffset   = HCF_SEND_MBX_CMD_BIT_NO;     
    /* Bitmask for Handshakeflags to ack recv. packet                                       */
    ptCommChannel->tRecvMbx.ulRecvACKBitmask    = HCF_RECV_MBX_ACK;            
    /* Bitnumber for recv packet ack flag (used for notification array indexing)            */
    ptCommChannel->tRecvMbx.bRecvACKBitoffset   = HCF_RECV_MBX_ACK_BIT_NO;     
    /* ------------------------------------------------------------------------------------ */

    /* Width of the handshake cell                                                          */
    ptCommChannel->bHandshakeWidth     = RCX_HANDSHAKE_SIZE_16BIT;             
    /* pointer to channels handshake cell                                                   */
    ptCommChannel->ptHandshakeCell     = (NETX_HANDSHAKE_CELL*)&(&ptDefaultDPM->HandshakeChannel.tCommFlags0)[iIdx]; 

    /* Pointer to channel's control block                                                   */
    ptCommChannel->ptControlBlock      = (NETX_CONTROL_BLOCK*)&ptDefaultDPM->CommChannel[iIdx].tControl; 
    /* Handshake bit associated with control block                                          */
    ptCommChannel->bControlBlockBit    = 0;                                    
    /* Size of the control block in bytes                                                   */
    ptCommChannel->ulControlBlockSize  = sizeof(NETX_CONTROL_BLOCK);           

    /* Pointer to channel's common status block                                             */
    ptCommChannel->ptCommonStatusBlock = (NETX_COMMON_STATUS_BLOCK*)&ptDefaultDPM->CommChannel[iIdx].tCommonStatus;   
    /* Handshake bit associated with Common status block                                    */
    ptCommChannel->bCommonStatusBit    = 0;                                    
    /* Size of the common status block in bytes                                             */
    ptCommChannel->ulCommonStatusSize  = sizeof(NETX_COMMON_STATUS_BLOCK);     

    /* Pointer to channel's extended status block                                           */
    ptCommChannel->ptExtendedStatusBlock = (NETX_EXTENDED_STATUS_BLOCK*)&ptDefaultDPM->CommChannel[iIdx].tExtendedStatus;  
    /* Handshake bit associated with Extended status block                                  */
    ptCommChannel->bExtendedStatusBit    = 0;                                  
    /* Size of the extended status block in bytes                                           */
    ptCommChannel->ulExtendedStatusSize  = sizeof(NETX_EXTENDED_STATUS_BLOCK); 
    ptCommChannel->pvInitMutex           = OS_CreateMutex();
    /* channel lock */
    ptCommChannel->pvLock                = OS_CreateLock();

    /* ------------------------------------------------------------------------------------ */
    /* Channel Input area                                                                   */
    /* ------------------------------------------------------------------------------------ */
    /* DPM Pointer to start of IO Instance                                                  */
    ptCommIn->pbDPMAreaStart       = ptDefaultDPM->CommChannel[iIdx].abPd0Input;         
    /* Length of IO Instance                                                                */
    ptCommIn->ulDPMAreaLength      = sizeof(ptDefaultDPM->CommChannel[iIdx].abPd0Input); 
    /* Handshake bit associated with IO Instance                                            */
    ptCommIn->bHandshakeBit        = HCF_PD0_IN_ACK_BIT_NO;                                  
    /* Handshake mode for this IO instance                                                  */
    ptCommIn->usHandshakeMode      = RCX_IO_MODE_BUFF_HST_CTRL;                              
    switch(ptCommIn->usHandshakeMode )
    {
      case RCX_IO_MODE_BUFF_DEV_CTRL:
        ptCommIn->bHandshakeBitState = RCX_FLAGS_NOT_EQUAL;
      break;
   
      case RCX_IO_MODE_BUFF_HST_CTRL:
        ptCommIn->bHandshakeBitState = RCX_FLAGS_EQUAL;
      break;

      default:
        /* Unknown or non handshake                                                         */
        ptCommIn->bHandshakeBitState = RCX_FLAGS_NONE;
      break; 
    }
    /* Synchronistation object                                                              */
    ptCommIn->pvMutex                = OS_CreateMutex();                                                   

    /* ------------------------------------------------------------------------------------ */
    /* Channel Output area                                                                  */
    /* ------------------------------------------------------------------------------------ */
    /* DPM Pointer to start of IO Instance                                                  */
    ptCommOut->pbDPMAreaStart      =  ptDefaultDPM->CommChannel[iIdx].abPd0Output;         
    /* Length of IO Instance                                                                */
    ptCommOut->ulDPMAreaLength      = sizeof(ptDefaultDPM->CommChannel[iIdx].abPd0Output); 
    /* Handshake bit associated with IO Instance                                            */
    ptCommOut->bHandshakeBit        = HCF_PD0_OUT_CMD_BIT_NO;                                  
    /* Handshake mode for this IO instance                                                  */
    ptCommOut->usHandshakeMode      = RCX_IO_MODE_BUFF_HST_CTRL;                               
    switch(ptCommOut->usHandshakeMode )
    {
      case RCX_IO_MODE_BUFF_DEV_CTRL:
        ptCommOut->bHandshakeBitState = RCX_FLAGS_NOT_EQUAL;
      break;
   
      case RCX_IO_MODE_BUFF_HST_CTRL:
        ptCommOut->bHandshakeBitState = RCX_FLAGS_EQUAL;
      break;

      default:
        /* Unknown or non handshake                                                         */
        ptCommOut->bHandshakeBitState = RCX_FLAGS_NONE;
      break; 
    }
    /* Synchronistation object                                                              */
    ptCommOut->pvMutex = OS_CreateMutex();           
  
    /* Number of Input areas                                                                */
    ptCommChannel->ulIOInputAreas      = 1;          
    ptCommChannel->pptIOInputAreas     = (PIOINSTANCE*)OS_Memalloc(sizeof(*ptCommChannel->pptIOInputAreas));
    /* Input Areas array for this channel                                                   */
    ptCommChannel->pptIOInputAreas[0]  = ptCommIn;   

    /* Number of Output areas                                                               */
    ptCommChannel->ulIOOutputAreas     = 1;         
    ptCommChannel->pptIOOutputAreas    = (PIOINSTANCE*)OS_Memalloc(sizeof(*ptCommChannel->pptIOOutputAreas));
    /* Output Areas array for this channel                                                  */
    ptCommChannel->pptIOOutputAreas[0] = ptCommOut; 

    ptCommChannel->fIsChannel       = 1;

    DEV_ReadHostFlags(ptCommChannel, 1);
    DEV_ReadHandshakeFlags(ptCommChannel, 0, 0);

    ptCommChannel->ulOpenCount      = 1;

    ++ptDevInst->ulCommChannelCount;
  }

  return lRet;
}

/*****************************************************************************/
/*! Initialize interrupt handling
 * \param ptDevInst          Buffer to the device instance structure
 * \param ulCommChannelCount Number of communication channels to init
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
void DEV_InitializeInterrupt(PDEVICEINSTANCE ptDevInst)
{
  /* the following initialization is only required if interrupt is enabled */
  /* Ask for interrupt handling */
  if(0 != ptDevInst->fIrqEnabled)
  {
    PCHANNELINSTANCE ptChannelInst = &ptDevInst->tSystemDevice;
    uint32_t         ulChannel     = 0;
    uint32_t         ulSync;

    /* create all synch events */
    for(ulSync = 0; ulSync < sizeof(ptDevInst->tSyncData.ahSyncBitEvents) / sizeof(ptDevInst->tSyncData.ahSyncBitEvents[0]); ++ulSync)
      ptDevInst->tSyncData.ahSyncBitEvents[ulSync] = OS_CreateEvent();

    /* Create events for all channels */
    do
    {
      uint32_t ulHandshakeWidth = 8;
      uint32_t ulIdx            = 0;

      /* Create interrupt events if we are working in interrupt mode */
      if(ptChannelInst->bHandshakeWidth == RCX_HANDSHAKE_SIZE_16BIT)
      {
        ulHandshakeWidth = 16;
      }

      for(ulIdx = 0; ulIdx < ulHandshakeWidth; ++ulIdx)
        ptChannelInst->ahHandshakeBitEvents[ulIdx] = OS_CreateEvent();

      /* Check if we have such a channel */
      if(ulChannel < ptDevInst->ulCommChannelCount)
        ptChannelInst = ptDevInst->pptCommChannels[ulChannel];

      /* Note: Check for <= as we are additionally evaluating the system channel */
    } while(ulChannel++ < ptDevInst->ulCommChannelCount);

    /* Perform a dummy interrupt cycle to get handshake flags in Sync for proper operation */
    if(CIFX_TKIT_IRQ_DSR_REQUESTED == cifXTKitISRHandler(ptDevInst, 1))
      cifXTKitDSRHandler(ptDevInst);

    OS_EnableInterrupts(ptDevInst->pvOSDependent);

    /* Set interrupt enable bits in PCI mode only if the complete 64KByte DPM is available */
    if (ptDevInst->ptGlobalRegisters)
    {
      /* Enable global and handshake interrupts */
      HWIF_WRITE32(ptDevInst, ptDevInst->ptGlobalRegisters->ulIRQEnable_0, 
                  HOST_TO_LE32((MSK_IRQ_EN0_INT_REQ | MSK_IRQ_EN0_HANDSHAKE) ));

      HWIF_WRITE32(ptDevInst, ptDevInst->ptGlobalRegisters->ulIRQEnable_1, 0);
    }
  }
}

/*****************************************************************************/
/*! Uninitialize interrupt handling
 * \param ptDevInst          Buffer to the device instance structure
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
void DEV_UninitializeInterrupt(PDEVICEINSTANCE ptDevInst)
{
  /* Check if interrupt is used */
  if(0 != ptDevInst->fIrqEnabled)
  {
    PCHANNELINSTANCE ptChannelInst = &ptDevInst->tSystemDevice;  /* Initialize ptChannel with system device */
    uint32_t         ulChannel     = 0;
    uint32_t         ulSync;

    /* disable hardware interrupt */
    OS_DisableInterrupts( ptDevInst->pvOSDependent);

    /* Remove channel events for all channels / start with system device */
    do
    {
      uint32_t ulHandshakeWidth = 8;
      uint32_t ulIdx            = 0;

      /* Create interrupt events if we are working in interrupt mode */
      if(ptChannelInst->bHandshakeWidth == RCX_HANDSHAKE_SIZE_16BIT)
      {
        ulHandshakeWidth = 16;
      }

      for(ulIdx = 0; ulIdx < ulHandshakeWidth; ++ulIdx)
      {
        if ( NULL != ptChannelInst->ahHandshakeBitEvents[ulIdx])
          OS_DeleteEvent( ptChannelInst->ahHandshakeBitEvents[ulIdx]);
      }

      /* Check if we have such a channel */
      if(ulChannel < ptDevInst->ulCommChannelCount)
        ptChannelInst = ptDevInst->pptCommChannels[ulChannel];

      /* Note: Check for <= as we are additionally evaluating the system channel */
    } while(ulChannel++ < ptDevInst->ulCommChannelCount);


    /* Remove all synch events */
    for(ulSync = 0; ulSync < sizeof(ptDevInst->tSyncData.ahSyncBitEvents) / sizeof(ptDevInst->tSyncData.ahSyncBitEvents[0]); ++ulSync)
    {
      if( NULL != ptDevInst->tSyncData.ahSyncBitEvents[ulSync])
        OS_DeleteEvent( ptDevInst->tSyncData.ahSyncBitEvents[ulSync]);
    }
  }
}


/*****************************************************************************/
/*! Uninitialize internall data structures used with DEV functions                           
 * \param ptDevInst   Pointer to the device instance
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
void DEV_Uninitialize( PDEVICEINSTANCE ptDevInst)
{
  int iIdx = 0;

  /* Remove pointer from the communication channel structure */
  for (iIdx = 0; iIdx < (int)ptDevInst->ulCommChannelCount; iIdx++)
  {
    PCHANNELINSTANCE ptCommChannel = ptDevInst->pptCommChannels[iIdx];

    PIOINSTANCE      ptCommIn      = ptCommChannel->pptIOInputAreas[0];
    PIOINSTANCE      ptCommOut     = ptCommChannel->pptIOOutputAreas[0];

    if( NULL != ptCommIn->pvMutex)   OS_DeleteMutex( ptCommIn->pvMutex);
    OS_Memfree( ptCommIn);

    if( NULL != ptCommOut->pvMutex)  OS_DeleteMutex( ptCommOut->pvMutex);
    OS_Memfree( ptCommOut);


    if( NULL != ptCommChannel->pvLock)      OS_DeleteLock ( ptCommChannel->pvLock);
    if( NULL != ptCommChannel->pvInitMutex) OS_DeleteMutex( ptCommChannel->pvInitMutex); 
    

    OS_Memfree( ptCommChannel->pptIOOutputAreas);
    OS_Memfree( ptCommChannel->pptIOInputAreas);

    OS_Memfree( ptCommChannel);
  }

  OS_Memfree( ptDevInst->pptCommChannels);
}
